﻿(function () {
    var payload = null;
    var map = null;
    var userMarker = null;
    var userLabel = null;     // ← NUEVO
    var markers = [];
    var labelOverlays = [];
    var activeInfo = null; // InfoWindow reutilizable

    // --- Config opcional de iconos (puedes usar tu SVG/PNG) ---
    function makePinIcon(url, w, h, opts) {
        opts = opts || {};
        return {
            url: url,
            size: new google.maps.Size(w, h),
            scaledSize: new google.maps.Size(w, h),
            anchor: opts.anchor || new google.maps.Point(w / 2, h), // pin: centro abajo
        };
    }
    // var ICONS = { data: makePinIcon("/img/pin.svg", 32, 48), user: makePinIcon("/img/user.svg", 28, 28, { anchor: new google.maps.Point(14,14) }) };
    var ICONS = null; // usa el pin por defecto si no definís iconos

    // ---- Toasts (para “copiado”) ----
    function ensureToastHost() {
        var host = document.querySelector('.toast-container');
        if (!host) {
            host = document.createElement('div');
            host.className = 'toast-container';
            document.body.appendChild(host);
        }
        return host;
    }
    function showToast(message) {
        var host = ensureToastHost();
        var el = document.createElement('div');
        el.className = 'toast';
        el.textContent = message;
        host.appendChild(el);
        setTimeout(function () {
            if (el && el.parentNode) el.parentNode.removeChild(el);
        }, 1800);
    }

    // ---- Utilidades ----
    function openHelpModal() {
        var m = document.getElementById('helpModal');
        var b = document.getElementById('helpBackdrop');
        var p = document.getElementById('helpPanel');
        if (!m || !b || !p) return;

        // mostrar contenedor (aún transparente)
        m.classList.remove('invisible', 'pointer-events-none');
        m.setAttribute('aria-hidden', 'false');
        document.body.style.overflow = 'hidden';

        // siguiente frame → iniciar transición a visible
        requestAnimationFrame(function () {
            m.classList.add('opacity-100');
            b.classList.add('opacity-100');
            p.classList.remove('opacity-0', 'scale-95', 'translate-y-2');
            p.classList.add('opacity-100', 'scale-100', 'translate-y-0');
        });
    }

    function closeHelpModal() {
        var m = document.getElementById('helpModal');
        var b = document.getElementById('helpBackdrop');
        var p = document.getElementById('helpPanel');
        if (!m || !b || !p) return;

        // iniciar transición a oculto
        m.classList.remove('opacity-100');
        b.classList.remove('opacity-100');
        p.classList.remove('opacity-100', 'scale-100', 'translate-y-0');
        p.classList.add('opacity-0', 'scale-95', 'translate-y-2');

        // al terminar (200ms) desactivar interacciones y ocultar
        setTimeout(function () {
            m.classList.add('invisible', 'pointer-events-none');
            m.setAttribute('aria-hidden', 'true');
            document.body.style.overflow = '';
        }, 200);
    }

    function wireHelpModal() {
        var m = document.getElementById('helpModal');
        if (!m) return;
        var backdrop = document.getElementById('helpBackdrop');
        var btnClose = document.getElementById('helpClose');
        var btnOk = document.getElementById('helpOk');

        [backdrop, btnClose, btnOk].forEach(function (el) {
            if (el) el.addEventListener('click', closeHelpModal);
        });

        window.addEventListener('keydown', function (e) {
            if (e.key === 'Escape' && !m.classList.contains('invisible')) closeHelpModal();
        });
    }


    function setupSearch(map) {
        var input = document.getElementById('searchBox');
        if (!input) return;

        input.placeholder = 'PAÍS, CIUDAD, CALLES — ej: "Paraguay, Asunción, Azara y Montevideo"';

        // Control nativo centrado
        var ctrl = document.createElement('div');
        ctrl.className = 'search-control';

        var inner = document.createElement('div');
        inner.className = 'search-inner';
        inner.appendChild(input);

        // Botón Buscar (Tailwind + tus reglas siguen aplicando si existen)
        var btn = document.createElement('button');
        btn.id = 'searchBtn';
        btn.className = 'search-btn px-4 py-2 bg-blue-600 text-white border border-blue-600 rounded-md shadow hover:bg-white hover:text-blue-600';
        btn.type = 'button';
        btn.textContent = 'Buscar';
        inner.appendChild(btn);

        // Botón Ayuda "?" circular (Tailwind)
        var help = document.createElement('button');
        help.id = 'helpBtn';
        help.className = 'help-btn w-10 h-10 rounded-full bg-blue-600 text-white font-extrabold border border-blue-600 shadow hover:bg-white hover:text-blue-600';
        help.type = 'button';
        help.title = 'Cómo buscar';
        help.textContent = '?';
        inner.appendChild(help);

        ctrl.appendChild(inner);
        map.controls[google.maps.ControlPosition.TOP_CENTER].push(ctrl);

        // Modal: listeners (una sola vez)
        wireHelpModal();

        // Click abrir modal
        help.addEventListener('click', openHelpModal);

        // ENTER → buscar normalizada
        input.addEventListener('keydown', function (ev) {
            if (ev.key !== 'Enter') return;
            ev.preventDefault();
            doSearchFromInput();
        });

        // Click Buscar
        btn.addEventListener('click', function () { doSearchFromInput(); });
    }


    // Centra el mapa, hace zoom y coloca/mueve el marcador del usuario + etiqueta
    function centerAndMark(lat, lng) {
        var pos = new google.maps.LatLng(lat, lng);
        map.panTo(pos);
        map.setZoom(16);

        if (userMarker) {
            userMarker.setPosition(pos);
        } else {
            var uOptions = { position: pos, map: map };
            if (ICONS && ICONS.user) uOptions.icon = ICONS.user;
            userMarker = new google.maps.Marker(uOptions);
        }

        if (userLabel && typeof userLabel.setPosition === 'function') {
            userLabel.setPosition(pos);
            userLabel.setText("Nueva ubicación");
        } else if (typeof LabelOverlay === 'function') {
            userLabel = new LabelOverlay(pos, "Nueva ubicación", { offsetX: 16, offsetY: 0 });
            userLabel.setMap(map);
        }

        // Notificar a C#
        try { window.chrome.webview.hostObjects.external.ReportCoordinates(lat, lng); } catch (_) { }
    }

    function escHtml(s) {
        return String(s || "")
            .replace(/&/g, "&amp;")
            .replace(/</g, "&lt;")
            .replace(/>/g, "&gt;")
            .replace(/"/g, "&quot;")
            .replace(/'/g, "&#39;");
    }
    function fmt(n) { return Number(n).toFixed(6); }
    function mapsUrl(lat, lng) { return "https://maps.google.com/?q=" + fmt(lat) + "," + fmt(lng); }

    function copyText(text) {
        return new Promise(function (resolve, reject) {
            if (navigator.clipboard && window.isSecureContext) {
                navigator.clipboard.writeText(text).then(resolve).catch(function () { legacyCopy(); });
            } else {
                legacyCopy();
            }
            function legacyCopy() {
                try {
                    var ta = document.createElement('textarea');
                    ta.value = text;
                    ta.style.position = 'fixed';
                    ta.style.top = '-9999px';
                    document.body.appendChild(ta);
                    ta.focus(); ta.select();
                    var ok = document.execCommand('copy');
                    document.body.removeChild(ta);
                    ok ? resolve() : reject();
                } catch (e) { reject(e); }
            }
        });
    }

    // ---- InfoWindow con acciones de compartir ----
    function infoContent(name, lat, lng) {
        var nameEsc = escHtml(name);
        var latStr = fmt(lat);
        var lngStr = fmt(lng);
        var url = mapsUrl(lat, lng);
        var msg = nameEsc + "%0A" + "Lat: " + latStr + "%0A" + "Lng: " + lngStr + "%0A" + encodeURIComponent(url);

        return ''
            + '<div class="iw">'
            + '  <h3 class="iw-head">' + nameEsc + '</h3>'
            + '  <div class="iw-meta">Lat: ' + latStr + '<br/>Lng: ' + lngStr + '</div>'

            + '  <div class="iw-actions">'
            + '    <button id="btnCopyCoords" class="btn">Copiar coords</button>'
            + '    <button id="btnCopyLink"   class="btn">Copiar link</button>'
            + '    <button id="btnOpenMaps"   class="btn" style="grid-column:1 / -1;">Abrir Maps</button>'
            + '    <button id="btnShareWA"    class="btn">WhatsApp</button>'
            + '    <button id="btnShareMail"  class="btn">Email</button>'
            + '  </div>'

            + '  <div class="iw-note">* El botón Guardar del sistema usará estas coordenadas.</div>'

            + '  <div hidden id="__data__"'
            + '       data-name="' + nameEsc + '"'
            + '       data-lat="' + latStr + '"'
            + '       data-lng="' + lngStr + '"'
            + '       data-url="' + escHtml(url) + '"'
            + '       data-msg="' + msg + '"></div>'
            + '</div>';
    }

    function wireInfoWindowHandlers() {
        var root = document.getElementById("__data__");
        if (!root) return;

        var name = root.getAttribute("data-name") || "";
        var lat = root.getAttribute("data-lat") || "";
        var lng = root.getAttribute("data-lng") || "";
        var url = root.getAttribute("data-url") || "";
        var msgEnc = root.getAttribute("data-msg") || "";

        var btnCopyCoords = document.getElementById("btnCopyCoords");
        var btnCopyLink = document.getElementById("btnCopyLink");
        var btnOpenMaps = document.getElementById("btnOpenMaps");
        var btnShareWA = document.getElementById("btnShareWA");
        var btnShareMail = document.getElementById("btnShareMail");

        if (btnCopyCoords) btnCopyCoords.onclick = function () {
            copyText(lat + "," + lng)
                .then(function () { showToast("Coordenadas copiadas"); })
                .catch(function () { showToast("No se pudo copiar"); });
        };
        if (btnCopyLink) btnCopyLink.onclick = function () {
            copyText(url)
                .then(function () { showToast("Enlace copiado"); })
                .catch(function () { showToast("No se pudo copiar"); });
        };
        if (btnOpenMaps) btnOpenMaps.onclick = function () {
            try { window.open(url, "_blank"); } catch (_) { location.href = url; }
        };
        if (btnShareWA) btnShareWA.onclick = function () {
            var wa = "https://wa.me/?text=" + msgEnc;
            try { window.open(wa, "_blank"); } catch (_) { location.href = wa; }
        };
        if (btnShareMail) btnShareMail.onclick = function () {
            var mail = "mailto:?subject=" + encodeURIComponent("Ubicación - " + name) + "&body=" + msgEnc;
            location.href = mail;
        };
    }

    // ---- Bridge desde C# ----
    if (window.chrome && window.chrome.webview) {
        window.chrome.webview.addEventListener('message', function (e) {
            try { payload = JSON.parse(e.data); } catch (err) { payload = null; }
            if (!payload) return;
            ensureGoogleMaps(payload.apiKey).then(function () {
                initMap(payload);
            });
        });
    } else {
        console.warn("WebView2 bridge no disponible.");
    }

    // Cargar Google Maps dinámicamente con la API key recibida
    function ensureGoogleMaps(apiKey) {
        return new Promise(function (resolve) {
            if (window.google && window.google.maps) return resolve();
            var cbName = "__maps_cb_" + Date.now();
            window[cbName] = function () { resolve(); try { delete window[cbName]; } catch (_) { } };
            var s = document.createElement('script');
            s.src = "https://maps.googleapis.com/maps/api/js?key="
                + encodeURIComponent(apiKey)
                + "&callback=" + cbName; // ← si quieres, quita libraries=places
            s.async = true; s.defer = true;
            document.head.appendChild(s);
        });
    }

    function initMap(data) {
        document.body.classList.toggle('allow-marking', !!data.allowMarking);

        var defaultCenter = { lat: -25.2637, lng: -57.5759 }; // Asunción
        map = new google.maps.Map(document.getElementById('map'), {
            center: defaultCenter,
            zoom: 12
        });

        // === Definir LabelOverlay DESPUÉS de que Google Maps exista ===
        function LabelOverlay(position, text, options) {
            this.position = position;       // google.maps.LatLng
            this.text = text || "";
            this.map = null;
            this.div = null;
            this.offsetX = (options && options.offsetX) || 8;
            this.offsetY = (options && options.offsetY) || 0;
            this.className = (options && options.className) || "map-label";
        }
        LabelOverlay.prototype = new google.maps.OverlayView();
        LabelOverlay.prototype.onAdd = function () {
            this.div = document.createElement('div');
            this.div.className = this.className;
            this.div.textContent = this.text;
            this.getPanes().overlayLayer.appendChild(this.div);
        };
        LabelOverlay.prototype.draw = function () {
            if (!this.div) return;
            var proj = this.getProjection();
            if (!proj) return;
            var pixel = proj.fromLatLngToDivPixel(this.position);
            var el = this.div;
            el.style.left = Math.round(pixel.x + this.offsetX) + "px"; // derecha del pin
            var h = el.offsetHeight || 0;
            el.style.top = Math.round(pixel.y - (h / 2) + this.offsetY) + "px";
        };
        LabelOverlay.prototype.onRemove = function () {
            if (this.div && this.div.parentNode) this.div.parentNode.removeChild(this.div);
            this.div = null;
        };
        LabelOverlay.prototype.setMap = function (m) {
            google.maps.OverlayView.prototype.setMap.call(this, m);
        };
        LabelOverlay.prototype.setPosition = function (latLng) {
            this.position = latLng;
            this.draw(); // reubica si ya está en el mapa
        };
        LabelOverlay.prototype.setText = function (text) {
            this.text = String(text || "");
            if (this.div) this.div.textContent = this.text;
        };

        // === fin LabelOverlay ===

        if (Array.isArray(data.points) && data.points.length > 0) {
            var bounds = new google.maps.LatLngBounds();

            data.points.forEach(function (p) {
                var mOptions = { position: { lat: p.lat, lng: p.lng }, map: map };
                if (ICONS && ICONS.data) mOptions.icon = ICONS.data;
                var m = new google.maps.Marker(mOptions);

                // dentro de initMap, al crear cada label:
                var label = new LabelOverlay(
                    new google.maps.LatLng(p.lat, p.lng),
                    String(p.name || ""),
                    { offsetX: 16, offsetY: 0 }   // ← más separación a la derecha del pin
                );
                label.setMap(map);
                labelOverlays.push(label);

                // click → InfoWindow + notificar a C#
                m.addListener('click', function () {
                    try { window.chrome.webview.hostObjects.external.ReportPoint(String(p.name || ""), p.lat, p.lng); } catch (_) { }

                    if (!activeInfo) activeInfo = new google.maps.InfoWindow();
                    activeInfo.setContent(infoContent(p.name || "", p.lat, p.lng));
                    activeInfo.open(map, m);
                    google.maps.event.addListenerOnce(activeInfo, 'domready', wireInfoWindowHandlers);
                });

                markers.push(m);
                bounds.extend(m.getPosition());
            });

            map.fitBounds(bounds);
            if (data.points.length === 1) map.setZoom(16);
        }

        if (data.allowMarking) {
            map.addListener('click', function (e) {
                var loc = e.latLng;

                // mover/crear el pin del usuario
                if (userMarker) {
                    userMarker.setPosition(loc);
                } else {
                    var uOptions = { position: loc, map: map };
                    if (ICONS && ICONS.user) uOptions.icon = ICONS.user;
                    userMarker = new google.maps.Marker(uOptions);
                }

                // mover/crear la etiqueta “Nueva marca aquí” a la DERECHA del pin
                if (userLabel) {
                    userLabel.setPosition(loc);
                    userLabel.setText("Nueva ubicación");
                } else {
                    userLabel = new LabelOverlay(loc, "Nueva ubicación", { offsetX: 16, offsetY: 0 });
                    userLabel.setMap(map);
                }

                // Notificar a C#
                try { window.chrome.webview.hostObjects.external.ReportCoordinates(loc.lat(), loc.lng()); } catch (_) { }

                // Cerrar InfoWindow activo si lo hubiera
                if (activeInfo) { try { activeInfo.close(); } catch (_) { } }
            });

            setupSearch(map);
        }
    }

    // -------- helpers de normalización por comas --------
    function sanitizeInput(s) {
        s = (s || "").replace(/\s+/g, " ").trim();
        if (!s || /^,+$/.test(s)) return "";
        return s;
    }

    // quita tildes para comparar país
    function stripDiacritics(s) {
        return (s || "").normalize("NFD").replace(/[\u0300-\u036f]/g, "");
    }

    // Devuelve { query, specifiedCountry } siguiendo tus reglas:
    // 0 comas  -> "Paraguay, Asunción, <texto>"
    // 1 coma   -> "Paraguay, <texto>"
    // >=2 comas-> respeta tal cual y toma la 1ª parte como país (si existe)
    function buildQuery(raw) {
        var s = sanitizeInput(raw);
        if (!s) return { query: "", specifiedCountry: null };

        var commaCount = (s.match(/,/g) || []).length;

        if (commaCount === 0) {
            return { query: "Paraguay, Asunción, " + s, specifiedCountry: null };
        }
        if (commaCount === 1) {
            return { query: "Paraguay, " + s, specifiedCountry: null };
        }

        // >=2 comas: país + ciudad + calles (o más)
        var firstPart = s.split(",")[0].trim();
        var country = firstPart || null; // tal cual lo escribió el usuario
        return { query: s, specifiedCountry: country };
    }

    // -------- geocode sesgado a Paraguay sólo si NO se indicó país --------
    var PY_BOUNDS = new google.maps.LatLngBounds(
        new google.maps.LatLng(-27.6, -63.7),
        new google.maps.LatLng(-19.2, -54.1)
    );

    function doSearchFromInput() {
        var input = document.getElementById('searchBox');
        if (!input) return;

        var built = buildQuery(input.value);
        if (!built.query) {
            try { showToast("Escribe una dirección válida"); } catch (_) { }
            return;
        }

        var geocoder = new google.maps.Geocoder();

        // si el usuario escribió un país y NO es Paraguay -> NO restringimos
        var countryRaw = built.specifiedCountry;
        var countryNorm = stripDiacritics((countryRaw || "")).toLowerCase();
        var isParaguay = countryNorm === "paraguay";

        var request = { address: built.query };

        if (!countryRaw) {
            // no indicó país (0 o 1 coma): sesgamos a PY
            request.bounds = PY_BOUNDS;
            request.region = "py";
            request.componentRestrictions = { country: "py" };
        } else if (isParaguay) {
            // indicó explícitamente Paraguay: también sesgamos a PY
            request.bounds = PY_BOUNDS;
            request.region = "py";
            request.componentRestrictions = { country: "py" };
        } // else: indicó otro país → sin restricciones

        geocoder.geocode(request, function (results, status) {
            if (status === "OK" && results && results.length) {
                // si no restringimos a PY, tomamos el primer resultado;
                // si restringimos a PY, priorizamos uno dentro de Paraguay
                var best = results[0];

                if (!countryRaw || isParaguay) {
                    best = results.find(function (r) {
                        return (r.address_components || []).some(function (c) {
                            return c.types && c.types.indexOf("country") >= 0 &&
                                /paraguay/i.test(c.long_name);
                        });
                    }) || results[0];
                }

                var loc = best.geometry.location;
                centerAndMark(loc.lat(), loc.lng());
            } else {
                try { showToast("No se encontró la ubicación"); } catch (_) { }
            }
        });
    }
})();
