(function () {
  const CFG  = window.APP_CONFIG || {};
  const DATA = window.APP_DATA   || {};
  const EDIT_ENABLED = (DATA.modo || 'UPD').toUpperCase() !== 'DEL';

  // === Carga dinámica de Google Maps (usa la key inyectada por C#) ===
  function loadGoogleMaps(key, onload){
    const s = document.createElement('script');
    s.src = 'https://maps.googleapis.com/maps/api/js?key=' + encodeURIComponent(key) + '&libraries=geometry';
    s.async = true; s.defer = true; s.onload = onload;
    document.head.appendChild(s);
  }

  document.addEventListener('DOMContentLoaded', () => {
    if(!CFG.googleMapsKey){ console.error('Falta APP_CONFIG.googleMapsKey'); GEO.alerta('No hay clave de Google Maps'); return; }
    loadGoogleMaps(CFG.googleMapsKey, initMap);
  });

  // === Estado del mapa / edición ===
  let map, polygonEdit, vertices=[], markers=[];
  let fraccionPolygon=null, manzanaPolygon=null;
  let fraccionLabels=[], manzanaLabels=[], loteLabels=[];
  let otrasFraccionPolys=[], otrasFraccionLabels=[];   // <-- NUEVO
  let otrasManzanaPolys=[], otrasManzanaLabels=[];
  let lotePolygons=[];

  // === API expuesta a WinForms (C# -> JS) ===
  function clearMap(){
    markers.forEach(m=>m.setMap(null)); markers=[];
    if(polygonEdit) polygonEdit.setMap(null);
    polygonEdit = new google.maps.Polygon({
      paths:[], strokeColor:'#FF0000', strokeOpacity:.8, strokeWeight:2, fillOpacity:0, map
    });
    vertices=[];
  }

  function deleteLast(){
    if(!vertices.length) return;
    vertices.pop(); polygonEdit.setPath(vertices);
    const m = markers.pop(); if(m) m.setMap(null);
    // Re-etiquetar
    markers.forEach((mk,i)=> {
      mk.setLabel({
        text:String(i+1),
        color:'#FFFFFF',
        fontSize:'12px',
        fontWeight:'bold'
      });
      mk.setZIndex(1000 + i + 1);
    });
  }

  function saveMap(){
    const tipo = (DATA.tipo || '').toLowerCase();

    // Validaciones duras antes de enviar
    if(tipo === 'lote'){
      if(!manzanaPolygon){ GEO.alerta('No existe geometría de manzana.'); return; }
      for(const v of vertices){
        if(!google.maps.geometry.poly.containsLocation(v, manzanaPolygon)){
          GEO.alerta('Hay puntos fuera de la manzana.'); return;
        }
      }
      if(lotePolygons.some(p => vertices.some(v => google.maps.geometry.poly.containsLocation(v, p)))){
        GEO.alerta('❌ El lote se superpone con otro existente.'); return;
      }
    } else if(tipo === 'manzana'){
      if(!fraccionPolygon){ GEO.alerta('No existe geometría de fracción.'); return; }
      for(const v of vertices){
        if(!google.maps.geometry.poly.containsLocation(v, fraccionPolygon)){
          GEO.alerta('Puntos fuera del límite de la fracción.'); return;
        }
      }
      if(otrasManzanaPolys.some(p => vertices.some(v => google.maps.geometry.poly.containsLocation(v, p)))){
        GEO.alerta('❌ La manzana se superpone con otra existente.'); return;
      }
    } else if(tipo === 'fraccion'){ // <-- NUEVO
      if(otrasFraccionPolys.some(p => vertices.some(v => google.maps.geometry.poly.containsLocation(v, p)))){
        GEO.alerta('❌ La fracción se superpone con otra fracción existente.'); return;
      }
    }

    const coords = vertices.map(v => ({ lat: v.lat(), lng: v.lng() }));
    if(window.chrome && window.chrome.webview){
      window.chrome.webview.postMessage(JSON.stringify(coords));
    }
  }

  // Exponer global
  window.clearMap   = clearMap;
  window.deleteLast = deleteLast;
  window.saveMap    = saveMap;

  // === Helpers de icono circular ===
  function makeCircleIcon(scale=12, fill='#1f7ae0', stroke='#ffffff'){
    return {
      path: google.maps.SymbolPath.CIRCLE,
      scale: scale,
      fillColor: fill,
      fillOpacity: 1,
      strokeColor: stroke,
      strokeWeight: 2,
      labelOrigin: new google.maps.Point(0, 0),
      anchor: new google.maps.Point(0, 0)
    };
  }

  // === Render principal ===
  function initMap(){
    map = new google.maps.Map(document.getElementById('map'), {
      center:{lat:-25.2637, lng:-57.5759},
      zoom:16,
      styles: GEO.MAP_STYLE,
      clickableIcons:false,
      disableDefaultUI:false,
      zoomControl:true
    });

    // Seguridad: si estos arrays no existen (según el tipo), créalos
    var otrasFraccionPolys   = window.otrasFraccionPolys   || (window.otrasFraccionPolys   = []);
    var otrasFraccionLabels  = window.otrasFraccionLabels  || (window.otrasFraccionLabels  = []);

    const tipo = (DATA.tipo || '').toLowerCase();

    // Fracción (contenedora) -> NO dibujar como contenedor si el tipo ES fracción (evita duplicar)
    if(tipo !== 'fraccion' && Array.isArray(DATA.fraccion) && DATA.fraccion.length>0){
      const pathF = GEO.latLngsFromArray(DATA.fraccion);
      fraccionPolygon = GEO.polygon(map, pathF, {
        strokeColor:'#072ac8', strokeOpacity:.7, strokeWeight:2,
        fillColor:'#e0ecff', fillOpacity:.2
      });
      if(DATA.nombreFraccion)
        fraccionLabels.push( GEO.crearEtiqueta(DATA.nombreFraccion, GEO.centro(pathF), map, true) );
    }

    // Manzana (contenedora) -> sólo para LOTE (en tipo manzana es el editable)
    if(tipo === 'lote' && Array.isArray(DATA.manzana) && DATA.manzana.length>0){
      const pathM = GEO.latLngsFromArray(DATA.manzana);
      manzanaPolygon = GEO.polygon(map, pathM, {
        strokeColor:'#007f5f', strokeOpacity:.85, strokeWeight:2, fillOpacity:.05
      });
      if(DATA.nombreManzana)
        manzanaLabels.push( GEO.crearEtiqueta(DATA.nombreManzana, GEO.centro(pathM), map, true) );
    }

    // Otras FRACCIONES (cuando llegue el arreglo desde FraccionMapBuilder)
    (DATA.otrasFracciones || []).forEach(f=>{
      const coords = GEO.latLngsFromArray(f.coordenadas || []);
      if(!coords.length) return;
      const p = GEO.polygon(map, coords, {
        strokeColor:'#072ac8', strokeOpacity:.7, strokeWeight:3, fillOpacity:.05
      });
      otrasFraccionPolys.push(p);
      if(f.nombre) otrasFraccionLabels.push( GEO.crearEtiqueta(f.nombre, GEO.centro(coords), map, true) );
    });

    // Otras MANZANAS (misma fracción)
    (DATA.otrasManzanas || []).forEach(mz=>{
      const coords = GEO.latLngsFromArray(mz.coordenadas || []);
      const p = GEO.polygon(map, coords, {
        strokeColor:'#2a9d8f', strokeOpacity:.5, strokeWeight:2, fillOpacity:.03
      });
      otrasManzanaPolys.push(p);
      if(mz.nombre) otrasManzanaLabels.push( GEO.crearEtiqueta(mz.nombre, GEO.centro(coords), map, false) );
    });

    // Otros LOTES (misma manzana)
    (DATA.otrosLotes || []).forEach(l=>{
      const coords = GEO.latLngsFromArray(l.coordenadas || []);
      const p = GEO.polygon(map, coords, {
        strokeColor:'#FFA500', strokeOpacity:.5, strokeWeight:2, fillOpacity:.05
      });
      lotePolygons.push(p);
      if(l.nombre) loteLabels.push( GEO.crearEtiqueta(l.nombre, GEO.centro(coords), map, false) );
    });

    // Polígono de edición
    polygonEdit = new google.maps.Polygon({
      paths:[], strokeColor:'#FF0000', strokeOpacity:.8, strokeWeight:2, fillOpacity:0, map
    });

    // Cargar trazado existente (con validaciones al cargar)
    loadTrace(DATA.geoData || []);

    // Etiquetar fracción ACTUAL tras cargar su traza (en modo "fraccion")
    if(tipo === 'fraccion' && DATA.nombreFraccion && vertices.length>0){
      fraccionLabels.push( GEO.crearEtiqueta(DATA.nombreFraccion, GEO.centro(vertices), map, true) );
    }

    // Edición por clic (según tipo y modo)
    if(EDIT_ENABLED){
      const clickHandler = (e)=>{
        const p = e.latLng;

        if(tipo === 'lote'){
          if(!manzanaPolygon){ GEO.alerta('No existe geometría de manzana.'); return; }
          if(!google.maps.geometry.poly.containsLocation(p, manzanaPolygon)){
            GEO.alerta('⚠️ El punto está fuera de la manzana.'); return;
          }
          if(lotePolygons.some(poly => google.maps.geometry.poly.containsLocation(p, poly))){
            GEO.alerta('❌ Ya existe un lote en esa ubicación.'); return;
          }
        } else if(tipo === 'manzana'){
          if(!fraccionPolygon){ GEO.alerta('No existe geometría de fracción.'); return; }
          if(!google.maps.geometry.poly.containsLocation(p, fraccionPolygon)){
            GEO.alerta('⚠️ El punto está fuera de la fracción.'); return;
          }
          if(otrasManzanaPolys.some(poly => google.maps.geometry.poly.containsLocation(p, poly))){
            GEO.alerta('❌ Ya existe una manzana en esta ubicación.'); return;
          }
        } else if(tipo === 'fraccion'){
          if(otrasFraccionPolys.some(poly => google.maps.geometry.poly.containsLocation(p, poly))){
            GEO.alerta('⚠️ No podés marcar dentro de otra fracción existente.'); return;
          }
        }

        addPoint(p);
      };
      map.addListener('click', clickHandler);
    } else {
      GEO.addBlockingOverlay();
      // No-op para las funciones de edición:
      window.clearMap = window.deleteLast = window.saveMap = function(){};
    }

    // --- visibilidad centralizada de etiquetas según zoom ---
    const ZOOM_FRACCION_LABEL_MAX = 16; // <17
    const ZOOM_MANZANA_LABEL_MIN  = 17; // >=17
    const ZOOM_LOTE_LABEL_MIN     = 18; // >=18

    function updateLabelVisibility(){
      const z = map.getZoom();
      // Fracciones (actual + otras)
      [...fraccionLabels, ...otrasFraccionLabels].forEach(l => l.setVisible(z <= ZOOM_FRACCION_LABEL_MAX));
      // Manzanas (actual + otras)  ⬅️ incluye otras manzanas
      [...manzanaLabels, ...otrasManzanaLabels].forEach(l => l.setVisible(z >= ZOOM_MANZANA_LABEL_MIN));
      // Lotes
      loteLabels.forEach(l => l.setVisible(z >= ZOOM_LOTE_LABEL_MIN));
    }

    map.addListener('zoom_changed', updateLabelVisibility);
    updateLabelVisibility(); // estado inicial

    // Regla dura: si es LOTE y no hay manzana -> bloquear
    if(tipo==='lote' && !manzanaPolygon){
      GEO.alerta('No se puede editar el Lote: la manzana no existe.');
      GEO.addBlockingOverlay();
    }
  }


  function loadTrace(coords){
    clearMap();
    let skipped = 0;
    const tipo = (DATA.tipo || '').toLowerCase();

    (coords || []).forEach((c,i)=>{
      const ll = new google.maps.LatLng(c.lat, c.lng);

      if(tipo==='lote' && manzanaPolygon && !google.maps.geometry.poly.containsLocation(ll, manzanaPolygon)){
        skipped++; return;
      }
      if(tipo==='manzana' && fraccionPolygon && !google.maps.geometry.poly.containsLocation(ll, fraccionPolygon)){
        skipped++; return;
      }
      if(tipo==='fraccion' && otrasFraccionPolys.length &&
         otrasFraccionPolys.some(p => google.maps.geometry.poly.containsLocation(ll, p))){
        skipped++; return;
      }

      addPoint(ll);
    });

    if(vertices.length>0){
      const b = new google.maps.LatLngBounds();
      vertices.forEach(v => b.extend(v));
      map.fitBounds(b);
    }
    if(skipped>0) GEO.alerta('Se omitieron ' + skipped + ' punto(s) fuera del área válida.');
  }

  function addPoint(latLng){
    vertices.push(latLng);
    polygonEdit.setPath(vertices);

    // === Marcador circular con número ===
    const index = vertices.length;
    const marker = new google.maps.Marker({
      position: latLng,
      map,
      icon: makeCircleIcon(12, '#1f7ae0', '#ffffff'),
      label: {
        text: String(index),
        color: '#FFFFFF',
        fontSize: '12px',
        fontWeight: 'bold'
      },
      clickable: false,
      zIndex: 1000 + index
    });

    markers.push(marker);
  }
})();
